import SwiftUI

struct CuriosityModeView: View {
    @StateObject private var surveyViewModel = SurveyViewModel()
    @Environment(\.dismiss) private var dismiss
    @State private var showingSurvey = false
    @State private var showingResults = false
    @State private var showingConfirmation = false
    
    var body: some View {
        NavigationStack {
            ZStack {
                Color.black.ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 20) {
                        // Header
                        VStack(spacing: 10) {
                            Image(systemName: "brain.head.profile")
                                .font(.largeTitle)
                                .foregroundColor(.purple)
                            
                            Text("Curiosity Mode")
                                .font(.title2)
                                .fontWeight(.bold)
                                .foregroundColor(.white)
                            
                            Text("Experiment with different lifestyle scenarios without affecting your main data")
                                .font(.caption)
                                .foregroundColor(.secondary)
                                .multilineTextAlignment(.center)
                        }
                        .padding()
                        
                        // Current Status
                        if surveyViewModel.isInCuriosityMode {
                            CuriosityModeIndicator()
                        }
                        
                        // Action Buttons
                        VStack(spacing: 15) {
                            if !surveyViewModel.isInCuriosityMode {
                                Button("Start Experimenting") {
                                    surveyViewModel.enterCuriosityMode()
                                    showingSurvey = true
                                }
                                .buttonStyle(.borderedProminent)
                                .tint(.purple)
                            } else {
                                Button("Modify Scenario") {
                                    showingSurvey = true
                                }
                                .buttonStyle(.bordered)
                                .tint(.purple)
                                
                                Button("View Results") {
                                    showingResults = true
                                }
                                .buttonStyle(.bordered)
                                .tint(.blue)
                                
                                Button("Save as Real Data") {
                                    showingConfirmation = true
                                }
                                .buttonStyle(.borderedProminent)
                                .tint(.green)
                                
                                Button("Exit Curiosity Mode") {
                                    surveyViewModel.exitCuriosityMode()
                                }
                                .buttonStyle(.bordered)
                                .tint(.red)
                            }
                        }
                        .padding()
                        
                        // Information Section
                        VStack(alignment: .leading, spacing: 15) {
                            Text("How It Works")
                                .font(.headline)
                                .foregroundColor(.white)
                            
                            VStack(alignment: .leading, spacing: 10) {
                                InfoRow(icon: "questionmark.circle", 
                                       title: "Experiment Safely", 
                                       description: "Try different lifestyle answers")
                                
                                InfoRow(icon: "chart.line.uptrend.xyaxis", 
                                       title: "See Impact", 
                                       description: "View how changes affect predictions")
                                
                                InfoRow(icon: "shield.checkered", 
                                       title: "Protected Data", 
                                       description: "Your real data stays unchanged")
                                
                                InfoRow(icon: "arrow.uturn.backward", 
                                       title: "Easy Reset", 
                                       description: "Return to original data anytime")
                            }
                        }
                        .padding()
                        .background(Color.gray.opacity(0.1))
                        .cornerRadius(12)
                        .padding(.horizontal)
                    }
                }
            }
        }
        .navigationBarHidden(true)
        .sheet(isPresented: $showingSurvey) {
            CuriositySurveyView(viewModel: surveyViewModel)
        }
        .sheet(isPresented: $showingResults) {
            CuriosityResultsView(surveyData: surveyViewModel.surveyData)
        }
        .confirmationDialog("Save Curiosity Data", isPresented: $showingConfirmation) {
            Button("Save as Real Data") {
                surveyViewModel.saveCuriosityDataAsReal()
            }
            Button("Cancel", role: .cancel) { }
        } message: {
            Text("This will replace your real lifestyle data with the curiosity mode data. This action cannot be undone.")
        }
    }
}

struct CuriosityModeIndicator: View {
    @State private var isAnimating = false
    
    var body: some View {
        HStack {
            Image(systemName: "brain.head.profile")
                .foregroundColor(.purple)
                .scaleEffect(isAnimating ? 1.2 : 1.0)
                .animation(.easeInOut(duration: 1.0).repeatForever(autoreverses: true), value: isAnimating)
                .onAppear { isAnimating = true }
            
            VStack(alignment: .leading) {
                Text("Curiosity Mode Active")
                    .font(.subheadline)
                    .fontWeight(.semibold)
                    .foregroundColor(.purple)
                
                Text("Experimenting with scenarios")
                    .font(.caption2)
                    .foregroundColor(.secondary)
            }
            
            Spacer()
        }
        .padding()
        .background(Color.purple.opacity(0.1))
        .cornerRadius(10)
        .padding(.horizontal)
    }
}


struct CuriositySurveyView: View {
    @ObservedObject var viewModel: SurveyViewModel
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            VStack {
                // Curiosity mode indicator at top
                HStack {
                    Image(systemName: "brain.head.profile")
                        .foregroundColor(.purple)
                    Text("Curiosity Mode")
                        .font(.caption)
                        .foregroundColor(.purple)
                    Spacer()
                    Button("Done") {
                        dismiss()
                    }
                }
                .padding()
                
                // Embed the regular survey view
                SurveyView()
                    .environmentObject(viewModel)
            }
        }
    }
}

struct CuriosityResultsView: View {
    let surveyData: SurveyData
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            ZStack {
                Color.black.ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 20) {
                        // Header
                        VStack {
                            Image(systemName: "chart.line.uptrend.xyaxis")
                                .font(.largeTitle)
                                .foregroundColor(.blue)
                            
                            Text("Scenario Results")
                                .font(.title2)
                                .fontWeight(.bold)
                                .foregroundColor(.white)
                            
                            Text("Based on your curiosity mode inputs")
                                .font(.caption)
                                .foregroundColor(.secondary)
                        }
                        .padding()
                        
                        // Risk Level Display
                        VStack(spacing: 10) {
                            Text("Overall Risk Level")
                                .font(.headline)
                                .foregroundColor(.white)
                            
                            Text(surveyData.overallRiskLevel.displayName)
                                .font(.title3)
                                .fontWeight(.bold)
                                .foregroundColor(colorForRiskLevel(surveyData.overallRiskLevel))
                                .padding()
                                .background(colorForRiskLevel(surveyData.overallRiskLevel).opacity(0.2))
                                .cornerRadius(10)
                        }
                        
                        // Lifestyle Summary
                        VStack(alignment: .leading, spacing: 15) {
                            Text("Lifestyle Summary")
                                .font(.headline)
                                .foregroundColor(.white)
                            
                            LifestyleSummaryRow(
                                icon: "smoke",
                                title: "Smoking",
                                value: surveyData.smokingStatus.displayName
                            )
                            
                            LifestyleSummaryRow(
                                icon: "lungs",
                                title: "Vaping",
                                value: surveyData.vapingStatus.displayName
                            )
                            
                            LifestyleSummaryRow(
                                icon: "wineglass",
                                title: "Alcohol",
                                value: "\(surveyData.alcoholUnitsPerWeek.formatted(.number.precision(.fractionLength(1)))) units/week"
                            )
                            
                            LifestyleSummaryRow(
                                icon: "forkknife",
                                title: "Fast Food",
                                value: "\(surveyData.fastFoodMealsPerWeek.formatted()) meals/week"
                            )
                            
                            LifestyleSummaryRow(
                                icon: "figure.walk",
                                title: "Exercise",
                                value: surveyData.exerciseFrequency.displayName
                            )
                        }
                        .padding()
                        .background(Color.gray.opacity(0.1))
                        .cornerRadius(12)
                        .padding(.horizontal)
                        
                        // Action Buttons
                        Button("Close") {
                            dismiss()
                        }
                        .buttonStyle(.borderedProminent)
                        .padding()
                    }
                }
            }
        }
        .navigationBarHidden(true)
    }
    
    private func colorForRiskLevel(_ level: RiskLevel) -> Color {
        switch level {
        case .low: return .green
        case .moderate: return .yellow
        case .high: return .orange
        case .veryHigh: return .red
        }
    }
}

struct LifestyleSummaryRow: View {
    let icon: String
    let title: String
    let value: String
    
    var body: some View {
        HStack {
            Image(systemName: icon)
                .foregroundColor(.blue)
                .frame(width: 20)
            
            Text(title)
                .foregroundColor(.white)
            
            Spacer()
            
            Text(value)
                .foregroundColor(.secondary)
                .font(.caption)
        }
    }
}

#Preview {
    CuriosityModeView()
}
